<?php
/**
 * Plugin Name: DM Mobile Location
 * Plugin URI: https://dependentmedia.com/dm-mobile-location/
 * Description: Track and display your food truck location. Update via iPhone app, display on your website.
 * Version: 2.2.1
 * Author: Dependent Media
 * Author URI: https://dependentmedia.com/
 * License: GPL v2 or later
 * License URI: https://www.gnu.org/licenses/gpl-2.0.html
 * Text Domain: dm-mobile-location
 */

// Prevent direct access
if (!defined('ABSPATH')) {
    exit;
}

class DM_Mobile_Location {

    const VERSION = '2.2.1';
    const OPTION_LOCATIONS = 'dm_mobile_location_locations';
    const OPTION_API_KEY_HASH = 'dm_mobile_location_api_key_hash';
    const OPTION_API_KEY_DISPLAY = 'dm_mobile_location_api_key_display';
    const OPTION_SETTINGS = 'dm_mobile_location_settings';
    const OPTION_RATE_LIMIT = 'dm_mobile_location_rate_limit';
    const OPTION_SECURITY_LOG = 'dm_mobile_location_security_log';

    private $default_settings = array(
        'marker_color' => '#e74c3c',
        'marker_icon' => 'default',
        'custom_marker_url' => '',
        'map_style' => 'openstreetmap',
        'popup_bg_color' => '#ffffff',
        'popup_text_color' => '#333333',
        'popup_show_address' => true,
        'popup_show_time' => true,
        'popup_show_directions' => true,
        'popup_custom_css' => '',
        'rate_limit_enabled' => true,
        'rate_limit_requests' => 60,
        'rate_limit_window' => 3600,
        'require_https' => false,
        'allowed_ips' => '',
    );

    private $map_styles = array(
        'openstreetmap' => array(
            'name' => 'OpenStreetMap (Default)',
            'url' => 'https://{s}.tile.openstreetmap.org/{z}/{x}/{y}.png',
            'attribution' => '© <a href="https://www.openstreetmap.org/copyright">OpenStreetMap</a> contributors'
        ),
        'cartodb_light' => array(
            'name' => 'CartoDB Positron (Light)',
            'url' => 'https://{s}.basemaps.cartocdn.com/light_all/{z}/{x}/{y}{r}.png',
            'attribution' => '© <a href="https://www.openstreetmap.org/copyright">OpenStreetMap</a> contributors © <a href="https://carto.com/attributions">CARTO</a>'
        ),
        'cartodb_dark' => array(
            'name' => 'CartoDB Dark Matter',
            'url' => 'https://{s}.basemaps.cartocdn.com/dark_all/{z}/{x}/{y}{r}.png',
            'attribution' => '© <a href="https://www.openstreetmap.org/copyright">OpenStreetMap</a> contributors © <a href="https://carto.com/attributions">CARTO</a>'
        ),
        'cartodb_voyager' => array(
            'name' => 'CartoDB Voyager',
            'url' => 'https://{s}.basemaps.cartocdn.com/rastertiles/voyager/{z}/{x}/{y}{r}.png',
            'attribution' => '© <a href="https://www.openstreetmap.org/copyright">OpenStreetMap</a> contributors © <a href="https://carto.com/attributions">CARTO</a>'
        ),
        'esri_worldstreet' => array(
            'name' => 'Esri World Street Map',
            'url' => 'https://server.arcgisonline.com/ArcGIS/rest/services/World_Street_Map/MapServer/tile/{z}/{y}/{x}',
            'attribution' => 'Tiles © Esri'
        ),
        'esri_satellite' => array(
            'name' => 'Esri Satellite',
            'url' => 'https://server.arcgisonline.com/ArcGIS/rest/services/World_Imagery/MapServer/tile/{z}/{y}/{x}',
            'attribution' => 'Tiles © Esri'
        ),
    );

    private $marker_icons = array(
        'default' => 'Default Pin',
        'food_truck' => 'Food Truck',
        'restaurant' => 'Restaurant',
        'star' => 'Star',
        'custom' => 'Custom Image'
    );

    public function __construct() {
        add_action('rest_api_init', array($this, 'register_rest_routes'));
        add_action('admin_menu', array($this, 'add_admin_menu'));
        add_action('admin_init', array($this, 'admin_init'));
        add_action('admin_enqueue_scripts', array($this, 'admin_enqueue_scripts'));
        add_shortcode('dm_mobile_location', array($this, 'render_shortcode'));
        register_activation_hook(__FILE__, array($this, 'activate'));
    }

    public function activate() {
        if (!get_option(self::OPTION_API_KEY_HASH)) {
            $this->regenerate_api_key();
        }
        if (!get_option(self::OPTION_LOCATIONS)) {
            update_option(self::OPTION_LOCATIONS, array());
        }
        if (!get_option(self::OPTION_SETTINGS)) {
            update_option(self::OPTION_SETTINGS, $this->default_settings);
        }
        // Migrate old single location to new format
        $this->migrate_legacy_location();
    }

    private function migrate_legacy_location() {
        $old_location = get_option('dm_mobile_location_data');
        if ($old_location && isset($old_location['lat']) && $old_location['lat'] !== null) {
            $locations = get_option(self::OPTION_LOCATIONS, array());
            if (empty($locations)) {
                $id = 'loc_' . uniqid();
                $locations[$id] = array(
                    'id' => $id,
                    'lat' => $old_location['lat'],
                    'lng' => $old_location['lng'],
                    'label' => $old_location['label'] ?? '',
                    'address' => $old_location['address'] ?? '',
                    'name' => 'Primary Location',
                    'marker_color' => '',
                    'is_primary' => true,
                    'updated' => $old_location['updated'] ?? gmdate('c')
                );
                update_option(self::OPTION_LOCATIONS, $locations);
            }
        }
        // Migrate old API key
        $old_key = get_option('dm_mobile_location_api_key');
        if ($old_key && !get_option(self::OPTION_API_KEY_DISPLAY)) {
            update_option(self::OPTION_API_KEY_DISPLAY, $old_key);
            update_option(self::OPTION_API_KEY_HASH, password_hash($old_key, PASSWORD_DEFAULT));
        }
    }

    private function regenerate_api_key() {
        $plain_key = 'dm_' . bin2hex(random_bytes(24));
        $hashed_key = password_hash($plain_key, PASSWORD_DEFAULT);
        update_option(self::OPTION_API_KEY_HASH, $hashed_key);
        update_option(self::OPTION_API_KEY_DISPLAY, $plain_key);
        return $plain_key;
    }

    private function verify_api_key_secure($provided_key) {
        $stored_hash = get_option(self::OPTION_API_KEY_HASH);
        $stored_display = get_option(self::OPTION_API_KEY_DISPLAY);
        
        if (!empty($stored_hash) && password_verify($provided_key, $stored_hash)) {
            return true;
        }
        if (!empty($stored_display) && hash_equals($stored_display, $provided_key)) {
            return true;
        }
        return false;
    }

    private function check_rate_limit($ip) {
        $settings = $this->get_settings();
        if (!$settings['rate_limit_enabled']) {
            return true;
        }
        $rate_data = get_transient(self::OPTION_RATE_LIMIT . '_' . md5($ip));
        if ($rate_data === false) {
            set_transient(self::OPTION_RATE_LIMIT . '_' . md5($ip), array('count' => 1, 'start' => time()), $settings['rate_limit_window']);
            return true;
        }
        if ($rate_data['count'] >= $settings['rate_limit_requests']) {
            $this->log_security_event('rate_limit_exceeded', $ip);
            return false;
        }
        $rate_data['count']++;
        set_transient(self::OPTION_RATE_LIMIT . '_' . md5($ip), $rate_data, $settings['rate_limit_window'] - (time() - $rate_data['start']));
        return true;
    }

    private function log_security_event($event_type, $ip, $details = '') {
        $log = get_option(self::OPTION_SECURITY_LOG, array());
        if (count($log) >= 100) { array_shift($log); }
        $log[] = array('time' => current_time('mysql'), 'event' => $event_type, 'ip' => $ip, 'details' => $details);
        update_option(self::OPTION_SECURITY_LOG, $log);
    }

    private function get_settings() {
        return wp_parse_args(get_option(self::OPTION_SETTINGS, array()), $this->default_settings);
    }

    private function get_client_ip() {
        if (!empty($_SERVER['HTTP_CLIENT_IP'])) { return sanitize_text_field($_SERVER['HTTP_CLIENT_IP']); }
        if (!empty($_SERVER['HTTP_X_FORWARDED_FOR'])) { return sanitize_text_field(explode(',', $_SERVER['HTTP_X_FORWARDED_FOR'])[0]); }
        if (!empty($_SERVER['REMOTE_ADDR'])) { return sanitize_text_field($_SERVER['REMOTE_ADDR']); }
        return '';
    }

    public function register_rest_routes() {
        $namespace = 'dm-mobile-location/v1';
        register_rest_route($namespace, '/locations', array('methods' => 'GET', 'callback' => array($this, 'get_locations'), 'permission_callback' => '__return_true'));
        register_rest_route($namespace, '/location/(?P<id>[a-zA-Z0-9_-]+)', array('methods' => 'GET', 'callback' => array($this, 'get_location_by_id'), 'permission_callback' => '__return_true'));
        register_rest_route($namespace, '/location', array('methods' => 'GET', 'callback' => array($this, 'get_primary_location'), 'permission_callback' => '__return_true'));
        register_rest_route($namespace, '/location', array('methods' => 'POST', 'callback' => array($this, 'update_location'), 'permission_callback' => array($this, 'verify_api_key'), 'args' => $this->get_location_args()));
        register_rest_route($namespace, '/location/(?P<id>[a-zA-Z0-9_-]+)', array('methods' => 'POST', 'callback' => array($this, 'update_location'), 'permission_callback' => array($this, 'verify_api_key'), 'args' => $this->get_location_args()));
        register_rest_route($namespace, '/location/(?P<id>[a-zA-Z0-9_-]+)', array('methods' => 'DELETE', 'callback' => array($this, 'delete_location'), 'permission_callback' => array($this, 'verify_api_key')));
    }

    private function get_location_args() {
        return array(
            'lat' => array('required' => true, 'type' => 'number', 'validate_callback' => function($v) { return is_numeric($v) && $v >= -90 && $v <= 90; }),
            'lng' => array('required' => true, 'type' => 'number', 'validate_callback' => function($v) { return is_numeric($v) && $v >= -180 && $v <= 180; }),
            'label' => array('required' => false, 'type' => 'string', 'sanitize_callback' => 'sanitize_text_field'),
            'address' => array('required' => false, 'type' => 'string', 'sanitize_callback' => 'sanitize_text_field'),
            'name' => array('required' => false, 'type' => 'string', 'sanitize_callback' => 'sanitize_text_field'),
            'marker_color' => array('required' => false, 'type' => 'string', 'sanitize_callback' => 'sanitize_hex_color'),
            'is_primary' => array('required' => false, 'type' => 'boolean')
        );
    }

    public function verify_api_key($request) {
        $ip = $this->get_client_ip();
        $settings = $this->get_settings();
        if ($settings['require_https'] && !is_ssl()) {
            $this->log_security_event('https_required', $ip);
            return new WP_Error('rest_forbidden', 'HTTPS required', array('status' => 403));
        }
        if (!empty($settings['allowed_ips'])) {
            $allowed = array_map('trim', explode("\n", $settings['allowed_ips']));
            if (!in_array($ip, $allowed)) {
                $this->log_security_event('ip_not_allowed', $ip);
                return new WP_Error('rest_forbidden', 'IP not allowed', array('status' => 403));
            }
        }
        if (!$this->check_rate_limit($ip)) {
            return new WP_Error('rate_limit_exceeded', 'Rate limit exceeded', array('status' => 429));
        }
        $provided_key = $request->get_header('X-API-Key');
        if (empty($provided_key) || !$this->verify_api_key_secure($provided_key)) {
            $this->log_security_event('invalid_api_key', $ip, substr($provided_key ?? '', 0, 10) . '...');
            return new WP_Error('rest_forbidden', 'Invalid API key', array('status' => 403));
        }
        $this->log_security_event('successful_auth', $ip);
        return true;
    }

    public function get_locations($request) {
        return new WP_REST_Response(array('success' => true, 'data' => array_values(get_option(self::OPTION_LOCATIONS, array()))), 200);
    }

    public function get_location_by_id($request) {
        $locations = get_option(self::OPTION_LOCATIONS, array());
        $id = $request->get_param('id');
        if (!isset($locations[$id])) {
            return new WP_REST_Response(array('success' => false, 'message' => 'Location not found'), 404);
        }
        return new WP_REST_Response(array('success' => true, 'data' => $locations[$id]), 200);
    }

    public function get_primary_location($request) {
        $locations = get_option(self::OPTION_LOCATIONS, array());
        foreach ($locations as $loc) {
            if (!empty($loc['is_primary'])) {
                return new WP_REST_Response(array('success' => true, 'data' => $loc), 200);
            }
        }
        if (!empty($locations)) {
            return new WP_REST_Response(array('success' => true, 'data' => reset($locations)), 200);
        }
        return new WP_REST_Response(array('success' => true, 'data' => null, 'message' => 'No location set yet'), 200);
    }

    public function update_location($request) {
        $id = $request->get_param('id');
        $locations = get_option(self::OPTION_LOCATIONS, array());
        if (empty($id)) { $id = 'loc_' . uniqid(); }
        $is_primary = $request->get_param('is_primary');
        if ($is_primary) { foreach ($locations as $k => $v) { $locations[$k]['is_primary'] = false; } }
        if (empty($locations)) { $is_primary = true; }
        $location = array(
            'id' => $id,
            'lat' => floatval($request->get_param('lat')),
            'lng' => floatval($request->get_param('lng')),
            'label' => $request->get_param('label') ?? '',
            'address' => $request->get_param('address') ?? '',
            'name' => $request->get_param('name') ?? 'Location',
            'marker_color' => $request->get_param('marker_color') ?? '',
            'is_primary' => (bool) $is_primary,
            'updated' => gmdate('c')
        );
        $locations[$id] = $location;
        update_option(self::OPTION_LOCATIONS, $locations);
        return new WP_REST_Response(array('success' => true, 'message' => 'Location updated', 'data' => $location), 200);
    }

    public function delete_location($request) {
        $id = $request->get_param('id');
        $locations = get_option(self::OPTION_LOCATIONS, array());
        if (!isset($locations[$id])) {
            return new WP_REST_Response(array('success' => false, 'message' => 'Location not found'), 404);
        }
        unset($locations[$id]);
        update_option(self::OPTION_LOCATIONS, $locations);
        return new WP_REST_Response(array('success' => true, 'message' => 'Location deleted'), 200);
    }

    public function add_admin_menu() {
        add_menu_page('Mobile Location', 'Mobile Location', 'manage_options', 'dm-mobile-location', array($this, 'render_admin_page'), 'dashicons-location-alt', 30);
        add_submenu_page('dm-mobile-location', 'Locations', 'Locations', 'manage_options', 'dm-mobile-location', array($this, 'render_admin_page'));
        add_submenu_page('dm-mobile-location', 'Map Settings', 'Map Settings', 'manage_options', 'dm-mobile-location-settings', array($this, 'render_settings_page'));
        add_submenu_page('dm-mobile-location', 'Security', 'Security', 'manage_options', 'dm-mobile-location-security', array($this, 'render_security_page'));
        add_submenu_page('dm-mobile-location', 'Setup Guide', 'Setup Guide', 'manage_options', 'dm-mobile-location-help', array($this, 'render_help_page'));
    }

    public function admin_enqueue_scripts($hook) {
        if (strpos($hook, 'dm-mobile-location') === false) { return; }
        wp_enqueue_style('wp-color-picker');
        wp_enqueue_script('wp-color-picker');
        wp_enqueue_media();
    }

    public function admin_init() {
        if (isset($_POST['dm_regenerate_api_key']) && check_admin_referer('dm_mobile_location_security')) {
            $new_key = $this->regenerate_api_key();
            add_settings_error('dm_mobile_location', 'api_key_regenerated', 'API key regenerated. New key: <code>' . esc_html($new_key) . '</code> - Copy it now!', 'updated');
        }
        if (isset($_POST['dm_save_settings']) && check_admin_referer('dm_mobile_location_settings')) {
            $settings = array(
                'marker_color' => sanitize_hex_color($_POST['marker_color'] ?? '#e74c3c'),
                'marker_icon' => sanitize_text_field($_POST['marker_icon'] ?? 'default'),
                'custom_marker_url' => esc_url_raw($_POST['custom_marker_url'] ?? ''),
                'map_style' => sanitize_text_field($_POST['map_style'] ?? 'openstreetmap'),
                'popup_bg_color' => sanitize_hex_color($_POST['popup_bg_color'] ?? '#ffffff'),
                'popup_text_color' => sanitize_hex_color($_POST['popup_text_color'] ?? '#333333'),
                'popup_show_address' => isset($_POST['popup_show_address']),
                'popup_show_time' => isset($_POST['popup_show_time']),
                'popup_show_directions' => isset($_POST['popup_show_directions']),
                'popup_custom_css' => wp_strip_all_tags($_POST['popup_custom_css'] ?? ''),
                'rate_limit_enabled' => isset($_POST['rate_limit_enabled']),
                'rate_limit_requests' => absint($_POST['rate_limit_requests'] ?? 60),
                'rate_limit_window' => absint($_POST['rate_limit_window'] ?? 3600),
                'require_https' => isset($_POST['require_https']),
                'allowed_ips' => sanitize_textarea_field($_POST['allowed_ips'] ?? ''),
            );
            update_option(self::OPTION_SETTINGS, $settings);
            add_settings_error('dm_mobile_location', 'settings_saved', 'Settings saved.', 'updated');
        }
        if (isset($_POST['dm_delete_location']) && check_admin_referer('dm_mobile_location_admin')) {
            $id = sanitize_text_field($_POST['location_id']);
            $locations = get_option(self::OPTION_LOCATIONS, array());
            if (isset($locations[$id])) { unset($locations[$id]); update_option(self::OPTION_LOCATIONS, $locations); }
            add_settings_error('dm_mobile_location', 'location_deleted', 'Location deleted.', 'updated');
        }
        if (isset($_POST['dm_set_primary']) && check_admin_referer('dm_mobile_location_admin')) {
            $id = sanitize_text_field($_POST['location_id']);
            $locations = get_option(self::OPTION_LOCATIONS, array());
            foreach ($locations as $k => $v) { $locations[$k]['is_primary'] = ($k === $id); }
            update_option(self::OPTION_LOCATIONS, $locations);
            add_settings_error('dm_mobile_location', 'primary_set', 'Primary location updated.', 'updated');
        }
        if (isset($_POST['dm_clear_security_log']) && check_admin_referer('dm_mobile_location_security')) {
            update_option(self::OPTION_SECURITY_LOG, array());
            add_settings_error('dm_mobile_location', 'log_cleared', 'Security log cleared.', 'updated');
        }
    }

    public function render_admin_page() {
        $locations = get_option(self::OPTION_LOCATIONS, array());
        $api_key = get_option(self::OPTION_API_KEY_DISPLAY);
        $api_url = rest_url('dm-mobile-location/v1/location');
        ?>
        <div class="wrap">
            <h1>Mobile Location</h1>
            <?php settings_errors('dm_mobile_location'); ?>
            <div class="card" style="max-width:600px;padding:20px;margin-bottom:20px;">
                <h2>API Configuration</h2>
                <table class="form-table">
                    <tr><th>API Endpoint</th><td><code style="background:#f0f0f0;padding:8px;display:block;word-break:break-all;"><?php echo esc_html($api_url); ?></code></td></tr>
                    <tr><th>API Key</th><td><code style="background:#f0f0f0;padding:8px;display:block;word-break:break-all;"><?php echo esc_html($api_key); ?></code><p class="description">Keep this secret!</p></td></tr>
                </table>
            </div>
            <div class="card" style="max-width:800px;padding:20px;margin-bottom:20px;">
                <h2>Locations</h2>
                <?php if (empty($locations)): ?>
                    <p><em>No locations yet. Use your iPhone app to add one.</em></p>
                <?php else: ?>
                    <table class="wp-list-table widefat fixed striped">
                        <thead><tr><th>Name</th><th>Address</th><th>Label</th><th>Updated</th><th>Primary</th><th>Actions</th></tr></thead>
                        <tbody>
                        <?php foreach ($locations as $id => $loc): ?>
                            <tr>
                                <td><strong><?php echo esc_html($loc['name'] ?? 'Location'); ?></strong></td>
                                <td><?php echo esc_html($loc['address'] ?? '-'); ?></td>
                                <td><?php echo esc_html($loc['label'] ?? '-'); ?></td>
                                <td><?php echo esc_html($loc['updated'] ?? '-'); ?></td>
                                <td><?php if (!empty($loc['is_primary'])): ?><span class="dashicons dashicons-star-filled" style="color:#f0c000;"></span><?php else: ?>
                                    <form method="post" style="display:inline;"><?php wp_nonce_field('dm_mobile_location_admin'); ?><input type="hidden" name="location_id" value="<?php echo esc_attr($id); ?>"><button type="submit" name="dm_set_primary" class="button button-small">Set Primary</button></form>
                                <?php endif; ?></td>
                                <td><form method="post" style="display:inline;"><?php wp_nonce_field('dm_mobile_location_admin'); ?><input type="hidden" name="location_id" value="<?php echo esc_attr($id); ?>"><button type="submit" name="dm_delete_location" class="button button-small" onclick="return confirm('Delete?');">Delete</button></form></td>
                            </tr>
                        <?php endforeach; ?>
                        </tbody>
                    </table>
                <?php endif; ?>
            </div>
            <div class="card" style="max-width:600px;padding:20px;">
                <h2>Shortcode</h2>
                <p><code>[dm_mobile_location]</code> - Show primary location</p>
                <p><code>[dm_mobile_location show="all"]</code> - Show all locations</p>
                <p><code>[dm_mobile_location id="loc_xxx"]</code> - Show specific location</p>
                <p><code>[dm_mobile_location height="500px" zoom="12"]</code> - Custom options</p>
            </div>
        </div>
        <?php
    }

    public function render_settings_page() {
        $settings = $this->get_settings();
        ?>
        <div class="wrap">
            <h1>Map Settings</h1>
            <?php settings_errors('dm_mobile_location'); ?>
            <form method="post">
                <?php wp_nonce_field('dm_mobile_location_settings'); ?>
                <div class="card" style="max-width:700px;padding:20px;margin-bottom:20px;">
                    <h2>Marker</h2>
                    <table class="form-table">
                        <tr><th>Color</th><td><input type="text" name="marker_color" value="<?php echo esc_attr($settings['marker_color']); ?>" class="dm-color-picker"></td></tr>
                        <tr><th>Icon</th><td><select name="marker_icon" id="marker_icon"><?php foreach ($this->marker_icons as $k => $v): ?><option value="<?php echo esc_attr($k); ?>" <?php selected($settings['marker_icon'], $k); ?>><?php echo esc_html($v); ?></option><?php endforeach; ?></select></td></tr>
                        <tr id="custom_marker_row" style="<?php echo $settings['marker_icon'] !== 'custom' ? 'display:none;' : ''; ?>"><th>Custom Image</th><td><input type="text" name="custom_marker_url" id="custom_marker_url" value="<?php echo esc_attr($settings['custom_marker_url']); ?>" class="regular-text"><button type="button" class="button" id="upload_marker_btn">Upload</button><p class="description">32x32px PNG recommended</p></td></tr>
                    </table>
                </div>
                <div class="card" style="max-width:700px;padding:20px;margin-bottom:20px;">
                    <h2>Map Style</h2>
                    <table class="form-table">
                        <tr><th>Tiles</th><td><select name="map_style"><?php foreach ($this->map_styles as $k => $v): ?><option value="<?php echo esc_attr($k); ?>" <?php selected($settings['map_style'], $k); ?>><?php echo esc_html($v['name']); ?></option><?php endforeach; ?></select></td></tr>
                    </table>
                </div>
                <div class="card" style="max-width:700px;padding:20px;margin-bottom:20px;">
                    <h2>Popup</h2>
                    <table class="form-table">
                        <tr><th>Background</th><td><input type="text" name="popup_bg_color" value="<?php echo esc_attr($settings['popup_bg_color']); ?>" class="dm-color-picker"></td></tr>
                        <tr><th>Text Color</th><td><input type="text" name="popup_text_color" value="<?php echo esc_attr($settings['popup_text_color']); ?>" class="dm-color-picker"></td></tr>
                        <tr><th>Show Address</th><td><label><input type="checkbox" name="popup_show_address" <?php checked($settings['popup_show_address']); ?>> Display address</label></td></tr>
                        <tr><th>Show Time</th><td><label><input type="checkbox" name="popup_show_time" <?php checked($settings['popup_show_time']); ?>> Display update time</label></td></tr>
                        <tr><th>Show Directions</th><td><label><input type="checkbox" name="popup_show_directions" <?php checked($settings['popup_show_directions']); ?>> Display "Get Directions" link</label></td></tr>
                        <tr><th>Custom CSS</th><td><textarea name="popup_custom_css" rows="4" class="large-text code"><?php echo esc_textarea($settings['popup_custom_css']); ?></textarea></td></tr>
                    </table>
                </div>
                <p class="submit"><button type="submit" name="dm_save_settings" class="button button-primary">Save Settings</button></p>
            </form>
        </div>
        <script>jQuery(function($){$('.dm-color-picker').wpColorPicker();$('#marker_icon').on('change',function(){$('#custom_marker_row').toggle($(this).val()==='custom');});$('#upload_marker_btn').on('click',function(e){e.preventDefault();var f=wp.media({title:'Select Marker',button:{text:'Use'},multiple:false});f.on('select',function(){$('#custom_marker_url').val(f.state().get('selection').first().toJSON().url);});f.open();});});</script>
        <?php
    }

    public function render_security_page() {
        $settings = $this->get_settings();
        $log = get_option(self::OPTION_SECURITY_LOG, array());
        ?>
        <div class="wrap">
            <h1>Security</h1>
            <?php settings_errors('dm_mobile_location'); ?>
            <form method="post"><?php wp_nonce_field('dm_mobile_location_security'); ?>
                <div class="card" style="max-width:600px;padding:20px;margin-bottom:20px;">
                    <h2>API Key</h2>
                    <p>Regenerate if compromised.</p>
                    <button type="submit" name="dm_regenerate_api_key" class="button" onclick="return confirm('Regenerate key?');">Regenerate API Key</button>
                </div>
            </form>
            <form method="post"><?php wp_nonce_field('dm_mobile_location_settings'); ?>
                <div class="card" style="max-width:600px;padding:20px;margin-bottom:20px;">
                    <h2>Rate Limiting</h2>
                    <table class="form-table">
                        <tr><th>Enable</th><td><label><input type="checkbox" name="rate_limit_enabled" <?php checked($settings['rate_limit_enabled']); ?>> Limit requests per IP</label></td></tr>
                        <tr><th>Max Requests</th><td><input type="number" name="rate_limit_requests" value="<?php echo esc_attr($settings['rate_limit_requests']); ?>" min="1" class="small-text"> per window</td></tr>
                        <tr><th>Window (sec)</th><td><input type="number" name="rate_limit_window" value="<?php echo esc_attr($settings['rate_limit_window']); ?>" min="60" class="small-text"></td></tr>
                    </table>
                </div>
                <div class="card" style="max-width:600px;padding:20px;margin-bottom:20px;">
                    <h2>Access Control</h2>
                    <table class="form-table">
                        <tr><th>Require HTTPS</th><td><label><input type="checkbox" name="require_https" <?php checked($settings['require_https']); ?>> Only accept HTTPS</label></td></tr>
                        <tr><th>IP Whitelist</th><td><textarea name="allowed_ips" rows="3" class="regular-text"><?php echo esc_textarea($settings['allowed_ips']); ?></textarea><p class="description">One IP per line. Empty = all allowed.<br>Your IP: <code><?php echo esc_html($this->get_client_ip()); ?></code></p></td></tr>
                    </table>
                </div>
                <p class="submit"><button type="submit" name="dm_save_settings" class="button button-primary">Save Security Settings</button></p>
            </form>
            <div class="card" style="max-width:800px;padding:20px;">
                <h2>Security Log</h2>
                <?php if (empty($log)): ?><p><em>No events.</em></p><?php else: ?>
                <table class="wp-list-table widefat fixed striped"><thead><tr><th>Time</th><th>Event</th><th>IP</th><th>Details</th></tr></thead><tbody>
                <?php foreach (array_reverse($log) as $e): ?>
                    <tr><td><?php echo esc_html($e['time']); ?></td><td><?php 
                        $labels = array('successful_auth'=>'<span style="color:green;">✓ Success</span>','invalid_api_key'=>'<span style="color:red;">✗ Invalid Key</span>','rate_limit_exceeded'=>'<span style="color:orange;">⚠ Rate Limited</span>','https_required'=>'<span style="color:red;">✗ HTTPS Required</span>','ip_not_allowed'=>'<span style="color:red;">✗ IP Blocked</span>');
                        echo $labels[$e['event']] ?? esc_html($e['event']); ?></td><td><code><?php echo esc_html($e['ip']); ?></code></td><td><?php echo esc_html($e['details']); ?></td></tr>
                <?php endforeach; ?>
                </tbody></table>
                <form method="post" style="margin-top:15px;"><?php wp_nonce_field('dm_mobile_location_security'); ?><button type="submit" name="dm_clear_security_log" class="button">Clear Log</button></form>
                <?php endif; ?>
            </div>
        </div>
        <?php
    }

    public function render_help_page() {
        $api_url = rest_url('dm-mobile-location/v1/location');
        ?>
        <div class="wrap">
            <h1>Setup Guide</h1>
            
            <style>
                .dm-help-section { background: #fff; border: 1px solid #ccd0d4; border-radius: 4px; padding: 20px; margin-bottom: 20px; max-width: 800px; }
                .dm-help-section h2 { margin-top: 0; padding-bottom: 10px; border-bottom: 1px solid #eee; }
                .dm-help-section h3 { color: #1d2327; margin-top: 20px; }
                .dm-step { display: flex; margin-bottom: 15px; }
                .dm-step-number { background: #2271b1; color: #fff; width: 28px; height: 28px; border-radius: 50%; display: flex; align-items: center; justify-content: center; font-weight: bold; margin-right: 12px; flex-shrink: 0; }
                .dm-step-content { flex: 1; }
                .dm-code { background: #f0f0f0; padding: 12px; border-radius: 4px; font-family: monospace; font-size: 13px; overflow-x: auto; margin: 10px 0; }
                .dm-tip { background: #fff8e5; border-left: 4px solid #ffb900; padding: 12px; margin: 15px 0; }
                .dm-warning { background: #fcf0f1; border-left: 4px solid #d63638; padding: 12px; margin: 15px 0; }
                .dm-table { width: 100%; border-collapse: collapse; margin: 15px 0; }
                .dm-table th, .dm-table td { padding: 10px; text-align: left; border-bottom: 1px solid #eee; }
                .dm-table th { background: #f6f7f7; }
                .dm-table code { background: #f0f0f0; padding: 2px 6px; border-radius: 3px; }
            </style>

            <!-- Quick Start -->
            <div class="dm-help-section">
                <h2>🚀 Quick Start</h2>
                
                <div class="dm-step">
                    <div class="dm-step-number">1</div>
                    <div class="dm-step-content">
                        <strong>Copy your API credentials</strong><br>
                        Go to <a href="<?php echo admin_url('admin.php?page=dm-mobile-location-security'); ?>">Security</a> and copy your API Key.
                    </div>
                </div>
                
                <div class="dm-step">
                    <div class="dm-step-number">2</div>
                    <div class="dm-step-content">
                        <strong>Add the map to your website</strong><br>
                        Add this shortcode to any page or post:
                        <div class="dm-code">[dm_mobile_location]</div>
                    </div>
                </div>
                
                <div class="dm-step">
                    <div class="dm-step-number">3</div>
                    <div class="dm-step-content">
                        <strong>Configure the iPhone app</strong><br>
                        Open the app, go to Settings, and enter:
                        <div class="dm-code">API Endpoint: <?php echo esc_html($api_url); ?></div>
                        <div class="dm-code">API Key: (paste from Security page)</div>
                    </div>
                </div>
                
                <div class="dm-step">
                    <div class="dm-step-number">4</div>
                    <div class="dm-step-content">
                        <strong>Update your location</strong><br>
                        Tap on the map or use GPS, then tap "Update Location". Your website will show your new location!
                    </div>
                </div>
            </div>

            <!-- iPhone App Setup -->
            <div class="dm-help-section">
                <h2>📱 iPhone App Setup</h2>
                
                <h3>Requirements</h3>
                <ul>
                    <li>Mac with Xcode 15 or later</li>
                    <li>iPhone running iOS 17.0 or later</li>
                    <li>Apple ID (free account works)</li>
                </ul>

                <h3>Installation Steps</h3>
                
                <div class="dm-step">
                    <div class="dm-step-number">1</div>
                    <div class="dm-step-content">
                        <strong>Open the project</strong><br>
                        Unzip <code>DMMobileLocation.zip</code> and double-click <code>DMMobileLocation.xcodeproj</code>
                    </div>
                </div>
                
                <div class="dm-step">
                    <div class="dm-step-number">2</div>
                    <div class="dm-step-content">
                        <strong>Configure signing</strong><br>
                        Click the project in the sidebar → Signing & Capabilities → Check "Automatically manage signing" → Select your Team (Apple ID)
                    </div>
                </div>
                
                <div class="dm-step">
                    <div class="dm-step-number">3</div>
                    <div class="dm-step-content">
                        <strong>Build and run</strong><br>
                        Select your iPhone or a simulator at the top, then press <code>Cmd + R</code>
                    </div>
                </div>

                <div class="dm-tip">
                    <strong>💡 First time on a real iPhone?</strong><br>
                    After installing, go to iPhone Settings → General → VPN & Device Management → Trust your developer profile.
                </div>

                <div class="dm-warning">
                    <strong>⚠️ Free Developer Account Limitation</strong><br>
                    Apps expire after 7 days and need to be reinstalled from Xcode. A paid Apple Developer account ($99/year) removes this limitation.
                </div>
            </div>

            <!-- Shortcode Reference -->
            <div class="dm-help-section">
                <h2>📝 Shortcode Reference</h2>
                
                <table class="dm-table">
                    <thead>
                        <tr>
                            <th>Shortcode</th>
                            <th>Description</th>
                        </tr>
                    </thead>
                    <tbody>
                        <tr>
                            <td><code>[dm_mobile_location]</code></td>
                            <td>Display the primary location</td>
                        </tr>
                        <tr>
                            <td><code>[dm_mobile_location show="all"]</code></td>
                            <td>Display all locations on one map</td>
                        </tr>
                        <tr>
                            <td><code>[dm_mobile_location id="loc_xxx"]</code></td>
                            <td>Display a specific location by ID</td>
                        </tr>
                        <tr>
                            <td><code>[dm_mobile_location height="500px"]</code></td>
                            <td>Custom map height</td>
                        </tr>
                        <tr>
                            <td><code>[dm_mobile_location zoom="12"]</code></td>
                            <td>Custom zoom level (1-18)</td>
                        </tr>
                        <tr>
                            <td><code>[dm_mobile_location no_location_message="Coming soon!"]</code></td>
                            <td>Custom message when no location is set</td>
                        </tr>
                    </tbody>
                </table>

                <h3>All Options</h3>
                <div class="dm-code">[dm_mobile_location width="100%" height="400px" zoom="15" show="primary" no_location_message="Location coming soon!"]</div>
            </div>

            <!-- API Reference -->
            <div class="dm-help-section">
                <h2>🔌 API Reference</h2>
                
                <p>Your API endpoint: <code><?php echo esc_html($api_url); ?></code></p>

                <h3>Endpoints</h3>
                <table class="dm-table">
                    <thead>
                        <tr>
                            <th>Method</th>
                            <th>Endpoint</th>
                            <th>Description</th>
                            <th>Auth Required</th>
                        </tr>
                    </thead>
                    <tbody>
                        <tr>
                            <td><code>GET</code></td>
                            <td>/location</td>
                            <td>Get primary location</td>
                            <td>No</td>
                        </tr>
                        <tr>
                            <td><code>GET</code></td>
                            <td>/locations</td>
                            <td>Get all locations</td>
                            <td>No</td>
                        </tr>
                        <tr>
                            <td><code>POST</code></td>
                            <td>/location</td>
                            <td>Create/update location</td>
                            <td>Yes</td>
                        </tr>
                        <tr>
                            <td><code>DELETE</code></td>
                            <td>/location/{id}</td>
                            <td>Delete a location</td>
                            <td>Yes</td>
                        </tr>
                    </tbody>
                </table>

                <h3>Example: Get Current Location</h3>
                <div class="dm-code">curl "<?php echo esc_html($api_url); ?>"</div>

                <h3>Example: Update Location</h3>
                <div class="dm-code">curl -X POST "<?php echo esc_html($api_url); ?>" \
  -H "Content-Type: application/json" \
  -H "X-API-Key: YOUR_API_KEY" \
  -d '{"lat": 37.7749, "lng": -122.4194, "label": "Lunch spot!", "address": "123 Main St"}'</div>
            </div>

            <!-- Troubleshooting -->
            <div class="dm-help-section">
                <h2>🔧 Troubleshooting</h2>
                
                <h3>"Invalid API key" error in the app</h3>
                <ul>
                    <li>Make sure the API key matches exactly (no extra spaces)</li>
                    <li>Check that this plugin is activated</li>
                    <li>Try <a href="<?php echo admin_url('admin.php?page=dm-mobile-location-security'); ?>">regenerating the API key</a></li>
                </ul>

                <h3>"Connection failed" error</h3>
                <ul>
                    <li>Verify the API endpoint URL is correct</li>
                    <li>Make sure your site has a valid SSL certificate (https://)</li>
                    <li>Check that your site is accessible from the internet</li>
                </ul>

                <h3>Map not showing on website</h3>
                <ul>
                    <li>Verify the shortcode is correct: <code>[dm_mobile_location]</code></li>
                    <li>Make sure at least one location has been set from the app</li>
                    <li>Check for JavaScript errors in browser console</li>
                </ul>

                <h3>Location not updating</h3>
                <ul>
                    <li>Clear your website cache (if using a caching plugin)</li>
                    <li>Verify "Location updated" message appears in the app</li>
                    <li>Check the <a href="<?php echo admin_url('admin.php?page=dm-mobile-location-security'); ?>">Security Log</a> for any blocked requests</li>
                </ul>

                <h3>Xcode build errors</h3>
                <ul>
                    <li><strong>"Signing requires a development team"</strong> — Select your Team in Signing & Capabilities</li>
                    <li><strong>"No provisioning profile"</strong> — Check "Automatically manage signing"</li>
                    <li><strong>iOS version errors</strong> — Set Minimum Deployments to iOS 17.0 in General tab</li>
                </ul>
            </div>

            <!-- Tips -->
            <div class="dm-help-section">
                <h2>💡 Tips</h2>
                
                <ul>
                    <li><strong>Test your connection</strong> — Always tap "Test Connection" in the app after entering your API details</li>
                    <li><strong>Automatic address</strong> — The app automatically looks up the address when you place a pin</li>
                    <li><strong>Multiple trucks?</strong> — Use <code>show="all"</code> in the shortcode to display all locations on one map</li>
                    <li><strong>Custom styling</strong> — Add CSS in Map Settings to customize the popup appearance</li>
                    <li><strong>Security</strong> — Enable "Require HTTPS" and set an IP whitelist for extra protection</li>
                </ul>
            </div>

            <!-- Support -->
            <div class="dm-help-section">
                <h2>📬 Need Help?</h2>
                
                <p>
                    <strong>Online Help Center:</strong> <a href="https://dependentmedia.com/dm-mobile-location-help-center/" target="_blank">dependentmedia.com/dm-mobile-location-help-center</a><br>
                    <strong>Email Support:</strong> <a href="mailto:support@dependentmedia.com">support@dependentmedia.com</a><br>
                    <strong>Privacy Policy:</strong> <a href="https://dependentmedia.com/dm-mobile-location-privacy-policy/" target="_blank">View Privacy Policy</a>
                </p>
            </div>

            <p style="color: #666; font-size: 12px; margin-top: 30px;">
                DM Mobile Location v<?php echo self::VERSION; ?> · © <?php echo date('Y'); ?> Dependent Media · <a href="https://dependentmedia.com/dm-mobile-location/" target="_blank">dependentmedia.com/dm-mobile-location</a>
            </p>
        </div>
        <?php
    }

    public function render_shortcode($atts) {
        $settings = $this->get_settings();
        $locations = get_option(self::OPTION_LOCATIONS, array());
        $atts = shortcode_atts(array('width'=>'100%','height'=>'400px','zoom'=>'15','show'=>'primary','id'=>'','no_location_message'=>'Location coming soon!'), $atts);
        
        $show_locations = array();
        if (!empty($atts['id'])) {
            if (isset($locations[$atts['id']])) { $show_locations[] = $locations[$atts['id']]; }
        } elseif ($atts['show'] === 'all') {
            $show_locations = array_values($locations);
        } else {
            foreach ($locations as $loc) { if (!empty($loc['is_primary'])) { $show_locations[] = $loc; break; } }
            if (empty($show_locations) && !empty($locations)) { $show_locations[] = reset($locations); }
        }

        $has_locations = !empty($show_locations);
        $map_id = 'dm-map-' . uniqid();
        $map_style = $this->map_styles[$settings['map_style']] ?? $this->map_styles['openstreetmap'];

        ob_start();
        ?>
        <link rel="stylesheet" href="https://unpkg.com/leaflet@1.9.4/dist/leaflet.css" crossorigin=""/>
        <?php if (!empty($settings['popup_custom_css'])): ?><style><?php echo $settings['popup_custom_css']; ?></style><?php endif; ?>
        <style>.dm-popup .leaflet-popup-content-wrapper,.dm-popup .leaflet-popup-tip{background:<?php echo esc_attr($settings['popup_bg_color']); ?>;color:<?php echo esc_attr($settings['popup_text_color']); ?>;}</style>
        <div id="<?php echo esc_attr($map_id); ?>" style="width:<?php echo esc_attr($atts['width']); ?>;height:<?php echo esc_attr($atts['height']); ?>;border-radius:8px;overflow:hidden;">
            <?php if (!$has_locations): ?><div style="display:flex;align-items:center;justify-content:center;height:100%;background:#f0f0f0;color:#666;"><p><?php echo esc_html($atts['no_location_message']); ?></p></div><?php endif; ?>
        </div>
        <?php if ($has_locations): ?>
        <script src="https://unpkg.com/leaflet@1.9.4/dist/leaflet.js" crossorigin=""></script>
        <script>
        (function(){
            var locs=<?php echo json_encode($show_locations); ?>,s=<?php echo json_encode(array('marker_color'=>$settings['marker_color'],'marker_icon'=>$settings['marker_icon'],'custom_marker_url'=>$settings['custom_marker_url'],'popup_show_address'=>$settings['popup_show_address'],'popup_show_time'=>$settings['popup_show_time'],'popup_show_directions'=>$settings['popup_show_directions'])); ?>,z=<?php echo intval($atts['zoom']); ?>,tile=<?php echo json_encode(array('url'=>$map_style['url'],'attr'=>$map_style['attribution'])); ?>;
            var aLat=0,aLng=0;locs.forEach(function(l){aLat+=l.lat;aLng+=l.lng;});aLat/=locs.length;aLng/=locs.length;
            var map=L.map('<?php echo esc_js($map_id); ?>').setView([aLat,aLng],z);
            L.tileLayer(tile.url,{attribution:tile.attr}).addTo(map);
            function icon(c){var u,sz=[30,40],an=[15,40],pa=[0,-40];
                if(s.marker_icon==='custom'&&s.custom_marker_url){u=s.custom_marker_url;sz=[32,32];an=[16,32];pa=[0,-32];}
                else if(s.marker_icon==='food_truck'){var svg='<svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 640 512" fill="'+c+'"><path d="M48 0C21.5 0 0 21.5 0 48V368c0 26.5 21.5 48 48 48H64c0 53 43 96 96 96s96-43 96-96H384c0 53 43 96 96 96s96-43 96-96h32c17.7 0 32-14.3 32-32s-14.3-32-32-32V288 256 237.3c0-17-6.7-33.3-18.7-45.3L512 114.7c-12-12-28.3-18.7-45.3-18.7H416V48c0-26.5-21.5-48-48-48H48zM416 160h50.7L544 237.3V256H416V160zM112 416a48 48 0 1 1 96 0 48 48 0 1 1-96 0zm368-48a48 48 0 1 1 0 96 48 48 0 1 1 0-96z"/></svg>';u='data:image/svg+xml;base64,'+btoa(svg);sz=[40,32];an=[20,32];pa=[0,-32];}
                else if(s.marker_icon==='restaurant'){var svg='<svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 448 512" fill="'+c+'"><path d="M416 0C400 0 288 32 288 176V288c0 35.3 28.7 64 64 64h32V480c0 17.7 14.3 32 32 32s32-14.3 32-32V352 240 32c0-17.7-14.3-32-32-32zM64 16C64 7.8 57.9 1 49.7 .1S34.2 4.6 32.4 12.5L2.1 148.8C.7 155.1 0 161.5 0 167.9c0 45.9 35.1 83.6 80 87.7V480c0 17.7 14.3 32 32 32s32-14.3 32-32V255.6c44.9-4.1 80-41.8 80-87.7c0-6.4-.7-12.8-2.1-19.1L191.6 12.5c-1.8-8-9.3-13.3-17.4-12.4S160 7.8 160 16V150.2c0 5.4-4.4 9.8-9.8 9.8c-5.1 0-9.3-3.9-9.8-9L127.9 14.6C127.2 6.3 120.3 0 112 0s-15.2 6.3-15.9 14.6L83.7 151c-.5 5.1-4.7 9-9.8 9c-5.4 0-9.8-4.4-9.8-9.8V16z"/></svg>';u='data:image/svg+xml;base64,'+btoa(svg);sz=[30,40];an=[15,40];pa=[0,-40];}
                else if(s.marker_icon==='star'){var svg='<svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 576 512" fill="'+c+'"><path d="M316.9 18C311.6 7 300.4 0 288.1 0s-23.4 7-28.8 18L195 150.3 51.4 171.5c-12 1.8-22 10.2-25.7 21.7s-.7 24.2 7.9 32.7L137.8 329 113.2 474.7c-2 12 3 24.2 12.9 31.3s23 8 33.8 2.3l128.3-68.5 128.3 68.5c10.8 5.7 23.9 4.9 33.8-2.3s14.9-19.3 12.9-31.3L437.7 329 googletag541.8 225.9c8.6-8.5 11.7-21.2 7.9-32.7s-13.7-19.9-25.7-21.7L phases381.2 150.3 316.9 18z"/></svg>';u='data:image/svg+xml;base64,'+btoa(svg.replace('googletag','').replace('phases',''));sz=[32,32];an=[16,32];pa=[0,-32];}
                else{var svg='<svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 384 512" fill="'+c+'"><path d="M215.7 499.2C267 435 384 279.4 384 192C384 86 298 0 192 0S0 86 0 192c0 87.4 117 243 168.3 307.2c12.3 15.3 35.1 15.3 47.4 0zM192 128a64 64 0 1 1 0 128 64 64 0 1 1 0-128z"/></svg>';u='data:image/svg+xml;base64,'+btoa(svg);}
                return L.icon({iconUrl:u,iconSize:sz,iconAnchor:an,popupAnchor:pa});
            }
            var bounds=[];
            locs.forEach(function(l){
                var c=l.marker_color||s.marker_color,p='<strong>'+(l.label||l.name||'We are here!')+'</strong>';
                if(s.popup_show_address&&l.address)p+='<br>'+l.address;
                if(s.popup_show_time&&l.updated)p+='<br><small>Updated: '+new Date(l.updated).toLocaleString()+'</small>';
                if(s.popup_show_directions)p+='<br><a href="https://www.google.com/maps/dir/?api=1&destination='+l.lat+','+l.lng+'" target="_blank" rel="noopener" style="display:inline-block;margin-top:8px;padding:6px 12px;background:#4285f4;color:#fff;text-decoration:none;border-radius:4px;font-size:13px;">📍 Get Directions</a>';
                L.marker([l.lat,l.lng],{icon:icon(c)}).addTo(map).bindPopup(p,{className:'dm-popup'}).openPopup();
                bounds.push([l.lat,l.lng]);
            });
            if(bounds.length>1)map.fitBounds(bounds,{padding:[50,50]});
        })();
        </script>
        <?php endif;
        return ob_get_clean();
    }
}

new DM_Mobile_Location();
